<?php

namespace App\Http\Controllers;

use App\Models\UserModel;
use App\Models\UserInfoModel;
use App\Models\MonkType;
use App\Models\KudiModel;
use App\Services\TemplateExportService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class DashboardController extends Controller
{
    public function index(Request $request){
        // Get user from session
        $user = UserModel::find(session('user_id'));
        
        if (!$user) {
            return redirect()->route('login.form')->with('error', 'Please login first');
        }
        
        // Base query - count from user_info table directly
        $query = \App\Models\UserInfoModel::query();
        
        // If not super admin, filter by kudi
        if (!$user->hasRole('super_admin')) {
            if ($user->kudi_id) {
                $query->where('kudi_id', $user->kudi_id);
            }
        }
        
        // Apply kudi filter if provided (only for super admin)
        $selectedKudiId = $request->query('kudi_id');
        if ($user->hasRole('super_admin') && $selectedKudiId) {
            $query->where('kudi_id', $selectedKudiId);
        }
        
        // Get total members count
        $totalMembers = (clone $query)->count();
        
        // Get monk type IDs
        $monkTypeId = \App\Models\MonkType::where('name', 'ភិក្ខុ')->value('id');
        $noviceTypeId = \App\Models\MonkType::where('name', 'សាមណេរ')->value('id');
        
        // Count by monk type
        $monks = (clone $query)->where('monk_type_id', $monkTypeId)->count();
        $novices = (clone $query)->where('monk_type_id', $noviceTypeId)->count();
        
        // For super admin, get all kudis count
        if ($user && $user->hasRole('super_admin')) {
            $kudis = KudiModel::count();
            $allKudis = KudiModel::orderByRaw('CAST(kudi_number AS UNSIGNED)')->get();
            
            // Get members by kudi for the table
            $membersByKudiQuery = \App\Models\UserInfoModel::query();
            if ($selectedKudiId) {
                $membersByKudiQuery->where('kudi_id', $selectedKudiId);
            }
            $membersByKudi = $membersByKudiQuery
                ->join('kudi', 'user_info.kudi_id', '=', 'kudi.id')
                ->selectRaw('kudi.kudi_number, COUNT(*) as total')
                ->groupBy('kudi.id', 'kudi.kudi_number')
                ->orderByRaw('CAST(kudi.kudi_number AS UNSIGNED)')
                ->get()
                ->map(function ($item) {
                    return [
                        'kudi_number' => $item->kudi_number,
                        'total' => $item->total
                    ];
                });
        } else {
            $kudis = 1; // Regular admin sees only their assigned kudi
            $allKudis = KudiModel::where('id', $user->kudi_id)->get();
            
            // Get members by kudi for regular admin (only their kudi)
            $membersByKudi = \App\Models\UserInfoModel::where('kudi_id', $user->kudi_id)
                ->join('kudi', 'user_info.kudi_id', '=', 'kudi.id')
                ->selectRaw('kudi.kudi_number, COUNT(*) as total')
                ->groupBy('kudi.id', 'kudi.kudi_number')
                ->get()
                ->map(function ($item) {
                    return [
                        'kudi_number' => $item->kudi_number,
                        'total' => $item->total
                    ];
                });
        }

        // If it's an AJAX request, return JSON response
        if ($request->ajax()) {
            $view = view('admin.dashboard.partials.members_table', compact('membersByKudi'))->render();
            return response()->json([
                'success' => true,
                'html' => $view,
                'stats' => [
                    'total_members' => $totalMembers,
                    'monks' => $monks,
                    'novices' => $novices,
                    'kudis' => $kudis
                ]
            ]);
        }

        // Get members count by Kudi with filters
        $membersByKudiQuery = \App\Models\UserInfoModel::select('kudi_id', \DB::raw('count(*) as total'))
            ->when(!$user->hasRole('super_admin') && $user->kudi_id, function($q) use ($user) {
                return $q->where('kudi_id', $user->kudi_id);
            })
            ->whereNotNull('kudi_id');
        
        // Apply kudi filter
        if ($request->filled('kudi_id')) {
            $membersByKudiQuery->where('kudi_id', $request->kudi_id);
        }
        
        // Apply monk type filter
        if ($request->filled('monk_type')) {
            if ($request->monk_type === 'monk') {
                $membersByKudiQuery->where('monk_type_id', $monkTypeId);
            } elseif ($request->monk_type === 'novice') {
                $membersByKudiQuery->where('monk_type_id', $noviceTypeId);
            }
        }
        
        $membersByKudi = $membersByKudiQuery
            ->groupBy('kudi_id')
            ->get();
        
        // Load kudi details for each member count
        $kudiIds = $membersByKudi->pluck('kudi_id')->filter();
        $kudiDetails = KudiModel::whereIn('id', $kudiIds)->get()->keyBy('id');
        
        // Format the data for the view
        $membersByKudi = $membersByKudi->map(function ($item) use ($kudiDetails) {
            $kudi = $kudiDetails->get($item->kudi_id);
            return [
                'kudi_number' => $kudi ? $kudi->kudi_number : 'Unknown',
                'total' => $item->total
            ];
        })->when($request->filled('search'), function($collection) use ($request) {
            return $collection->filter(function($item) use ($request) {
                return stripos($item['kudi_number'], $request->search) !== false;
            });
        })
        ->sortBy('kudi_number')
        ->values();

        return view('admin.dashboard.index', compact(
            'totalMembers',
            'monks',
            'novices',
            'kudis',
            'allKudis',
            'membersByKudi'
        ));
    }
    
    public function export()
    {
        // Get user from session
        $user = null;
        if (session()->has('user_id')) {
            $user = UserModel::find(session('user_id'));
        }
        
        if (!$user) {
            abort(403, 'Unauthorized action.');
        }
        
        // Build query with relationships
        $query = UserInfoModel::with([
                'province', 
                'district', 
                'commune', 
                'village', 
                'currentProvince',
                'currentDistrict',
                'currentCommune',
                'currentVillage',
                'education', 
                'educationYear', 
                'monkType', 
                'kudi'
            ])
            ->leftJoin('kudi', 'user_info.kudi_id', '=', 'kudi.id')
            ->leftJoin('education_year', 'user_info.education_year_id', '=', 'education_year.id');
        
        // If not super admin, filter by their kudi
        if (!$user->hasRole('super_admin')) {
            if ($user->kudi_id) {
                $query->where('user_info.kudi_id', $user->kudi_id);
            } else {
                abort(403, 'No kudi assigned to your account.');
            }
        }
        
        $members = $query->orderByRaw('COALESCE(kudi.kudi_number, "ZZZZ") ASC')
            ->orderBy('user_info.name_kh', 'asc')
            ->select('user_info.*', 'kudi.kudi_number as kudi_number', 'education_year.name as education_year_name')
            ->get();
        
        $templateService = new TemplateExportService();
        $templatePath = storage_path('app/templates/Listmonk.xlsx');
        $filename = 'members_' . date('Y-m-d') . '.xlsx';
        
        return $templateService->copyTemplateAndFillData($members, $templatePath, $filename);
    }
}
