<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Illuminate\Support\Facades\Http;

class ApiVillageSeeder extends Seeder
{
    public function run()
    {
        // Clear existing villages
        DB::statement('SET FOREIGN_KEY_CHECKS=0');
        DB::table('village')->truncate();

        $created_at = Carbon::now();
        $allVillages = [];
        
        // First get total pages
        try {
            $response = Http::timeout(30)->get("https://data.mef.gov.kh/api/v1/public-datasets/pd_66a8603a00604c000123e147/json?page=1&page_size=10");
            $data = $response->json();
            $totalPages = $data['total_pages'] ?? 1459;
        } catch (\Exception $e) {
            $this->command->error("Failed to get total pages: " . $e->getMessage());
            return;
        }

        $this->command->info("Total pages to fetch: " . $totalPages);

        // Fetch all pages with better error handling
        for ($page = 1; $page <= $totalPages; $page++) {
            try {
                $url = "https://data.mef.gov.kh/api/v1/public-datasets/pd_66a8603a00604c000123e147/json?page={$page}&page_size=10";
                $response = Http::timeout(60)->get($url); // Increased timeout
                $data = $response->json();

                if (isset($data['items'])) {
                    foreach ($data['items'] as $item) {
                        $allVillages[] = [
                            'id' => (int)$item['village_code'],
                            'commune_id' => (int)$item['commune_code'],
                            'name' => $item['village_kh'],
                            'created_at' => $created_at,
                            'updated_at' => $created_at
                        ];
                    }
                    
                    // Progress update every 50 pages
                    if ($page % 50 == 0) {
                        $this->command->info("Fetched page {$page}/{$totalPages} - Total villages: " . count($allVillages));
                    }
                    
                    // Stop if we've reached the last page
                    if ($data['page'] >= $data['total_pages']) {
                        break;
                    }
                }
            } catch (\Exception $e) {
                $this->command->error("Failed to fetch page {$page}: " . $e->getMessage());
                // Continue with next page instead of breaking
                continue;
            }
            
            // Small delay to avoid overwhelming the API
            usleep(100000); // 0.1 second delay
        }

        $this->command->info("Finished fetching. Total villages collected: " . count($allVillages));

        // Insert in smaller batches with duplicate handling
        $batchSize = 500; // Reduced batch size
        $batches = array_chunk($allVillages, $batchSize);
        
        foreach ($batches as $index => $batch) {
            try {
                DB::table('village')->insert($batch);
                $this->command->info("Inserted batch " . ($index + 1) . "/" . count($batches) . " with " . count($batch) . " villages");
            } catch (\Illuminate\Database\QueryException $e) {
                if (strpos($e->getMessage(), 'Duplicate entry') !== false) {
                    $this->command->info("Skipping batch " . ($index + 1) . " due to duplicates");
                    // Insert individually to skip duplicates
                    $inserted = 0;
                    foreach ($batch as $village) {
                        try {
                            DB::table('village')->insert($village);
                            $inserted++;
                        } catch (\Illuminate\Database\QueryException $e2) {
                            if (strpos($e2->getMessage(), 'Duplicate entry') === false) {
                                throw $e2;
                            }
                        }
                    }
                    $this->command->info("Inserted batch " . ($index + 1) . " individually: {$inserted}/" . count($batch) . " villages");
                } else {
                    throw $e;
                }
            }
        }

        $this->command->info("Successfully inserted " . count($allVillages) . " villages with Khmer names!");
    }
}
