<!DOCTYPE html>
<html>
<head>
    <title>Debug Console</title>
    <style>
        body { font-family: Arial; padding: 20px; }
        .console-output { 
            background: #000; 
            color: #0f0; 
            padding: 10px; 
            height: 300px; 
            overflow-y: auto; 
            font-family: monospace; 
            margin: 10px 0; 
        }
        select { margin: 5px; padding: 5px; }
    </style>
</head>
<body>
    <h2>Debug Village Selection</h2>
    
    <form data-location-form>
        <label>Province: 
            <select data-role="province" data-endpoint="/api/provinces/:id/districts">
                <option value="">Select Province</option>
                <?php
                $provinces = \App\Models\Province::orderBy('name')->get();
                foreach($provinces as $province) {
                    echo "<option value='{$province->id}'>{$province->name}</option>";
                }
                ?>
            </select>
        </label><br><br>
        
        <label>District: 
            <select data-role="district" data-endpoint="/api/districts/:id/communes" disabled>
                <option value="">Select District</option>
            </select>
        </label><br><br>
        
        <label>Commune: 
            <select data-role="commune" data-endpoint="/api/communes/:id/villages" disabled>
                <option value="">Select Commune</option>
            </select>
        </label><br><br>
        
        <label>Village: 
            <select data-role="village" disabled>
                <option value="">Select Village</option>
            </select>
        </label>
    </form>
    
    <h3>Console Output:</h3>
    <div id="console-output" class="console-output"></div>
    
    <script>
    const consoleOutput = document.getElementById('console-output');
    
    // Override console.log to show on page
    const originalLog = console.log;
    console.log = function(...args) {
        originalLog.apply(console, args);
        consoleOutput.innerHTML += args.join(' ') + '\n';
        consoleOutput.scrollTop = consoleOutput.scrollHeight;
    };
    
    const originalError = console.error;
    console.error = function(...args) {
        originalError.apply(console, args);
        consoleOutput.innerHTML += 'ERROR: ' + args.join(' ') + '\n';
        consoleOutput.scrollTop = consoleOutput.scrollHeight;
    };
    
    (function() {
        console.log('Starting location script...');
        
        function initLocationSelects() {
            const form = document.querySelector('form[data-location-form]');
            if (!form) {
                console.log('No form found');
                return;
            }
            
            console.log('Form found');
            
            const provinceSelect = form.querySelector('[data-role="province"]');
            const districtSelect = form.querySelector('[data-role="district"]');
            const communeSelect = form.querySelector('[data-role="commune"]');
            const villageSelect = form.querySelector('[data-role="village"]');
            
            console.log('Selects found:', {
                province: !!provinceSelect,
                district: !!districtSelect,
                commune: !!communeSelect,
                village: !!villageSelect
            });
            
            async function loadOptions(select, url) {
                if (!select || !url) {
                    console.log('Missing select or URL');
                    return;
                }
                
                console.log('Loading options for:', url);
                select.disabled = true;
                select.innerHTML = '<option value="">Loading...</option>';
                
                try {
                    console.log('Fetching:', url);
                    const response = await fetch(url);
                    console.log('Response status:', response.status);
                    
                    if (!response.ok) {
                        throw new Error('HTTP ' + response.status);
                    }
                    
                    const data = await response.json();
                    console.log('Data received:', data);
                    console.log('Items count:', (data.data || []).length);
                    
                    const items = data.data || [];
                    select.innerHTML = '<option value="">Select</option>';
                    
                    items.forEach((item, index) => {
                        const option = document.createElement('option');
                        option.value = item.id;
                        option.textContent = item.name;
                        select.appendChild(option);
                        if (index < 3) console.log('Item', index, ':', item.id, item.name);
                    });
                    
                    select.disabled = false;
                    console.log('Options loaded successfully');
                } catch (error) {
                    console.error('Load error:', error);
                    select.innerHTML = '<option value="">Error loading</option>';
                    select.disabled = true;
                }
            }
            
            function resetSelect(select, label) {
                if (!select) return;
                console.log('Resetting', label);
                select.innerHTML = '<option value="">Select ' + label + '</option>';
                select.disabled = true;
            }
            
            if (provinceSelect && districtSelect) {
                provinceSelect.addEventListener('change', function() {
                    const value = this.value;
                    console.log('Province changed to:', value);
                    
                    resetSelect(communeSelect, 'Commune');
                    resetSelect(villageSelect, 'Village');
                    
                    if (value) {
                        loadOptions(districtSelect, `/api/provinces/${value}/districts`);
                    } else {
                        resetSelect(districtSelect, 'District');
                    }
                });
            }
            
            if (districtSelect && communeSelect) {
                districtSelect.addEventListener('change', function() {
                    const value = this.value;
                    console.log('District changed to:', value);
                    
                    resetSelect(villageSelect, 'Village');
                    
                    if (value) {
                        loadOptions(communeSelect, `/api/districts/${value}/communes`);
                    } else {
                        resetSelect(communeSelect, 'Commune');
                    }
                });
            }
            
            if (communeSelect && villageSelect) {
                communeSelect.addEventListener('change', function() {
                    const value = this.value;
                    console.log('Commune changed to:', value);
                    
                    if (value) {
                        loadOptions(villageSelect, `/api/communes/${value}/villages`);
                    } else {
                        resetSelect(villageSelect, 'Village');
                    }
                });
            }
            
            console.log('Event listeners attached');
        }
        
        document.addEventListener('DOMContentLoaded', function() {
            console.log('DOM loaded');
            initLocationSelects();
        });
    })();
    </script>
</body>
</html>
