/**
 * AJAX Kudi Filter - Fetches data from controller
 * Works on all pages with kudi filtering
 */

// Global filter object
window.KudiFilter = {
    init: function() {
        console.log('Kudi Filter: Starting initialization...');
        console.log('Kudi Filter: Current URL:', window.location.href);
        console.log('Kudi Filter: Pathname:', window.location.pathname);
        
        // Find the select element
        this.select = document.querySelector('select[name="kudi_id"]');
        if (!this.select) {
            console.error('Kudi Filter: No select element found');
            return false;
        }
        
        console.log('Kudi Filter: Select found with', this.select.options.length, 'options');
        
        // Find table body
        this.tableBody = document.querySelector('tbody');
        console.log('Kudi Filter: Table body found:', !!this.tableBody);
        
        if (!this.tableBody) {
            console.error('Kudi Filter: No table body found');
            // Try alternative selectors
            this.tableBody = document.querySelector('#user-table-body');
            console.log('Kudi Filter: Alternative table body found:', !!this.tableBody);
            
            if (!this.tableBody) {
                console.error('Kudi Filter: No table body found with any selector');
                return false;
            }
        }
        
        console.log('Kudi Filter: Table body element:', this.tableBody);
        console.log('Kudi Filter: Table body innerHTML length:', this.tableBody.innerHTML.length);
        
        // Prevent form submission that causes page refresh
        const form = this.select.closest('form');
        if (form) {
            console.log('Kudi Filter: Found form, preventing submission');
            form.addEventListener('submit', function(e) {
                console.log('Kudi Filter: Preventing form submission');
                e.preventDefault();
                e.stopPropagation();
                return false;
            });
        }
        
        // Add change event
        this.select.addEventListener('change', this.filter.bind(this));
        
        console.log('Kudi Filter: Initialization complete');
        return true;
    },
    
    filter: function() {
        const selectedValue = this.select.value;
        const selectedText = this.select.options[this.select.selectedIndex].textContent.trim();
        
        console.log('Kudi Filter: Filter function called');
        console.log('Kudi Filter: Selected value:', selectedValue);
        console.log('Kudi Filter: Selected text:', selectedText);
        console.log('Kudi Filter: Table body before:', this.tableBody);
        
        // Show loading state
        this.showLoading();
        
        // Get form action URL
        const form = this.select.closest('form');
        const url = form ? form.action : window.location.href;
        console.log('Kudi Filter: Form action URL:', url);
        
        // Build URL with kudi_id parameter
        const fetchUrl = new URL(url);
        if (selectedValue) {
            fetchUrl.searchParams.set('kudi_id', selectedValue);
        } else {
            fetchUrl.searchParams.delete('kudi_id');
        }
        
        console.log('Kudi Filter: Fetching from', fetchUrl.toString());
        
        // Get CSRF token
        const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '';
        console.log('Kudi Filter: CSRF token found:', !!csrfToken);
        
        // Make AJAX request
        console.log('Kudi Filter: Starting AJAX request...');
        fetch(fetchUrl.toString(), {
            method: 'GET',
            headers: {
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-TOKEN': csrfToken,
                'Accept': 'application/json, text/html',
            },
        })
        .then(response => {
            console.log('Kudi Filter: Response received');
            console.log('Kudi Filter: Response status:', response.status);
            console.log('Kudi Filter: Response headers:', response.headers);
            
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }
            
            const contentType = response.headers.get('content-type');
            console.log('Kudi Filter: Content type:', contentType);
            
            if (contentType && contentType.includes('application/json')) {
                // Handle JSON response
                return response.json().then(data => {
                    console.log('Kudi Filter: JSON response received:', data);
                    this.handleJSONResponse(data);
                });
            } else {
                // Handle HTML response
                return response.text().then(html => {
                    console.log('Kudi Filter: HTML response received, length:', html.length);
                    this.handleHTMLResponse(html);
                });
            }
        })
        .catch(error => {
            console.error('Kudi Filter: AJAX request failed:', error);
            console.error('Kudi Filter: Error details:', error.message);
            this.hideLoading();
            this.showError('Failed to load data. Please try again.');
        });
    },
    
    showLoading: function() {
        if (!this.tableBody) return;
        
        // Store original content
        this.originalContent = this.tableBody.innerHTML;
        
        // Show loading indicator with proper styling
        const loadingHTML = `
            <tr class="loading-row">
                <td colspan="20" class="text-center py-5">
                    <div class="d-flex flex-column align-items-center">
                        <div class="spinner-border text-primary mb-3" role="status" style="width: 3rem; height: 3rem;">
                            <span class="visually-hidden">Loading...</span>
                        </div>
                        <div class="text-muted">
                            <i class="fas fa-spinner fa-spin me-2"></i>
                            កំពុងផ្ទុកទិន្នន័យ...
                        </div>
                    </div>
                </td>
            </tr>
        `;
        this.tableBody.innerHTML = loadingHTML;
    },
    
    hideLoading: function() {
        if (!this.tableBody || !this.originalContent) return;
        this.tableBody.innerHTML = this.originalContent;
    },
    
    showError: function(message) {
        if (!this.tableBody) return;
        
        const errorHTML = `
            <tr class="error-row">
                <td colspan="20" class="text-center py-4">
                    <div class="error-state-container">
                        <i class="fas fa-exclamation-triangle fa-3x text-warning mb-3"></i>
                        <h5 class="text-muted mb-2">ប្រុងជាន់បញ្ហា</h5>
                        <p class="text-muted mb-2">${message}</p>
                        <small class="text-muted">សូមព្យាយាមម្ដងទៀត</small>
                    </div>
                </td>
            </tr>
            <style>
            .error-state-container {
                padding: 2rem 1rem;
                min-height: 150px;
                display: flex;
                flex-direction: column;
                align-items: center;
                justify-content: center;
            }
            </style>
        `;
        this.tableBody.innerHTML = errorHTML;
    },
    
    handleJSONResponse: function(data) {
        console.log('Kudi Filter: Processing JSON response');
        console.log('Kudi Filter: Response data:', data);
        
        if (data.success && data.html) {
            // Update table with new HTML from server
            this.tableBody.innerHTML = data.html;
            console.log('Kudi Filter: Table updated with server data');
        } else if (data.html) {
            // Handle cases where success might be false but HTML is provided
            this.tableBody.innerHTML = data.html;
            console.log('Kudi Filter: Table updated with fallback HTML');
        } else {
            console.error('Kudi Filter: Invalid JSON response format');
            this.showError('Invalid response from server');
        }
    },
    
    handleHTMLResponse: function(html) {
        console.log('Kudi Filter: Processing HTML response');
        console.log('Kudi Filter: HTML length:', html.length);
        
        // If HTML is empty or contains error, show error state
        if (!html || html.trim() === '') {
            console.error('Kudi Filter: Empty HTML response');
            this.showError('No data received from server');
            return;
        }
        
        // Parse HTML and extract table body
        const parser = new DOMParser();
        const doc = parser.parseFromString(html, 'text/html');
        
        const newTableBody = doc.querySelector('tbody');
        
        if (newTableBody) {
            this.tableBody.innerHTML = newTableBody.innerHTML;
            console.log('Kudi Filter: Table updated with HTML response');
        } else {
            console.error('Kudi Filter: Could not extract table body from HTML');
            // Try to use the HTML directly if it contains table rows
            if (html.includes('<tr>')) {
                this.tableBody.innerHTML = html;
                console.log('Kudi Filter: Table updated with direct HTML');
            } else {
                this.showError('Could not process server response');
            }
        }
    }
};

// Auto-initialize when DOM is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', function() {
        window.KudiFilter.init();
    });
} else {
    // DOM already loaded
    window.KudiFilter.init();
}
