@php
    $name = $name ?? 'time';
    $value = $value ?? null;
    $required = $required ?? false;
    $placeholder = $placeholder ?? 'ជ្រើសរើសម៉ោង';
    $disabled = $disabled ?? false;
    $class = $class ?? 'form-control';
@endphp

<!-- Khmer Time Picker Component -->
<label class="field">
    <span>{{ $label ?? 'ម៉ោង' }}{{ $required ? ' *' : '' }}</span>
    <div class="khmer-time-input-group">
        <input 
            type="text" 
            name="{{ $name }}" 
            value="{{ $value }}"
            placeholder="{{ $placeholder }}"
            class="{{ $class }} khmer-time-input"
            {{ $required ? 'required' : '' }}
            {{ $disabled ? 'disabled' : '' }}
            readonly
        >
        <button type="button" class="khmer-time-btn" {{ $disabled ? 'disabled' : '' }}>
            <i class="fas fa-clock"></i>
        </button>
    </div>
</label>

<!-- Khmer Time Picker Modal -->
<div class="khmer-time-modal" id="khmer-time-modal-{{ $name }}" style="display: none;">
    <div class="khmer-time-content">
        <div class="khmer-time-header">
            <h5>ជ្រើសរើសម៉ោង</h5>
            <button type="button" class="khmer-time-close">&times;</button>
        </div>
        <div class="khmer-time-body">
            <div class="khmer-time-selectors">
                <div class="khmer-time-selector">
                    <label>ម៉ោង</label>
                    <select class="khmer-hour" data-name="{{ $name }}">
                        @for ($i = 0; $i <= 23; $i++)
                            <option value="{{ str_pad($i, 2, '0', STR_PAD_LEFT) }}">{{ str_pad($i, 2, '0', STR_PAD_LEFT) }}</option>
                        @endfor
                    </select>
                </div>
                <div class="khmer-time-separator">:</div>
                <div class="khmer-time-selector">
                    <label>នាទី</label>
                    <select class="khmer-minute" data-name="{{ $name }}">
                        @for ($i = 0; $i <= 59; $i += 5)
                            <option value="{{ str_pad($i, 2, '0', STR_PAD_LEFT) }}">{{ str_pad($i, 2, '0', STR_PAD_LEFT) }}</option>
                        @endfor
                    </select>
                </div>
            </div>
        </div>
        <div class="khmer-time-footer">
            <button type="button" class="btn btn-secondary khmer-time-cancel">បោះបង់</button>
            <button type="button" class="btn btn-primary khmer-time-confirm" data-name="{{ $name }}">យល់ព្រម</button>
        </div>
    </div>
</div>

@push('scripts')
    @once
        <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Khmer Time Picker functionality
            const timePickers = document.querySelectorAll('.khmer-time-input-group');
            
            timePickers.forEach(function(pickerGroup) {
                const input = pickerGroup.querySelector('.khmer-time-input');
                const button = pickerGroup.querySelector('.khmer-time-btn');
                const name = input.name;
                const modal = document.getElementById('khmer-time-modal-' + name);
                
                if (!modal) return;
                
                const hourSelect = modal.querySelector('.khmer-hour');
                const minuteSelect = modal.querySelector('.khmer-minute');
                const confirmBtn = modal.querySelector('.khmer-time-confirm');
                const cancelBtn = modal.querySelector('.khmer-time-cancel');
                const closeBtn = modal.querySelector('.khmer-time-close');
                
                // Open modal
                function openModal() {
                    modal.style.display = 'flex';
                    
                    // Set current values
                    const currentValue = input.value;
                    if (currentValue && currentValue.includes(':')) {
                        const [hour, minute] = currentValue.split(':');
                        hourSelect.value = hour;
                        minuteSelect.value = minute;
                    } else {
                        // Default to current time
                        const now = new Date();
                        hourSelect.value = String(now.getHours()).padStart(2, '0');
                        minuteSelect.value = String(Math.floor(now.getMinutes() / 5) * 5).padStart(2, '0');
                    }
                }
                
                // Close modal
                function closeModal() {
                    modal.style.display = 'none';
                }
                
                // Set time
                function setTime() {
                    const hour = hourSelect.value;
                    const minute = minuteSelect.value;
                    const timeString = hour + ':' + minute;
                    input.value = timeString;
                    
                    // Trigger change event
                    const event = new Event('change', { bubbles: true });
                    input.dispatchEvent(event);
                    
                    closeModal();
                }
                
                // Event listeners
                button.addEventListener('click', openModal);
                input.addEventListener('click', openModal);
                
                confirmBtn.addEventListener('click', setTime);
                cancelBtn.addEventListener('click', closeModal);
                closeBtn.addEventListener('click', closeModal);
                
                // Close modal when clicking outside
                modal.addEventListener('click', function(e) {
                    if (e.target === modal) {
                        closeModal();
                    }
                });
                
                // Keyboard support
                document.addEventListener('keydown', function(e) {
                    if (e.key === 'Escape' && modal.style.display === 'flex') {
                        closeModal();
                    }
                });
            });
        });
        </script>
        
        <style>
        /* Khmer Time Picker Styles */
        .khmer-time-input-group {
            display: flex;
            align-items: stretch;
            gap: 0;
        }
        
        .khmer-time-input {
            border-top-right-radius: 0 !important;
            border-bottom-right-radius: 0 !important;
            cursor: pointer;
        }
        
        .khmer-time-btn {
            border-top-left-radius: 0 !important;
            border-bottom-left-radius: 0 !important;
            border: 1px solid #ced4da;
            border-left: none;
            background: #f8f9fa;
            padding: 0 12px;
            cursor: pointer;
            transition: all 0.2s ease;
        }
        
        .khmer-time-btn:hover:not(:disabled) {
            background: #e9ecef;
            border-color: #adb5bd;
        }
        
        .khmer-time-btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }
        
        /* Modal Styles */
        .khmer-time-modal {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 9999;
            font-family: 'Kantumruy Pro', system-ui, sans-serif;
        }
        
        .khmer-time-content {
            background: white;
            border-radius: 12px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.3);
            max-width: 400px;
            width: 90%;
            max-height: 90vh;
            overflow: hidden;
            animation: slideUp 0.3s ease;
        }
        
        @keyframes slideUp {
            from {
                opacity: 0;
                transform: translateY(20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .khmer-time-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .khmer-time-header h5 {
            margin: 0;
            font-size: 18px;
            font-weight: 600;
        }
        
        .khmer-time-close {
            background: none;
            border: none;
            color: white;
            font-size: 24px;
            cursor: pointer;
            padding: 0;
            width: 30px;
            height: 30px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            transition: background 0.2s ease;
        }
        
        .khmer-time-close:hover {
            background: rgba(255, 255, 255, 0.2);
        }
        
        .khmer-time-body {
            padding: 30px 20px;
        }
        
        .khmer-time-selectors {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 15px;
        }
        
        .khmer-time-selector {
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 8px;
        }
        
        .khmer-time-selector label {
            font-size: 14px;
            font-weight: 500;
            color: #495057;
        }
        
        .khmer-time-selector select {
            width: 80px;
            padding: 10px;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            font-size: 16px;
            text-align: center;
            font-family: 'Kantumruy Pro', system-ui, sans-serif;
            background: white;
            cursor: pointer;
            transition: border-color 0.2s ease;
        }
        
        .khmer-time-selector select:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        .khmer-time-separator {
            font-size: 24px;
            font-weight: bold;
            color: #495057;
            margin-top: 20px;
        }
        
        .khmer-time-footer {
            padding: 20px;
            border-top: 1px solid #e9ecef;
            display: flex;
            justify-content: flex-end;
            gap: 10px;
            background: #f8f9fa;
        }
        
        .khmer-time-footer .btn {
            padding: 8px 20px;
            border-radius: 6px;
            font-size: 14px;
            font-weight: 500;
            font-family: 'Kantumruy Pro', system-ui, sans-serif;
            border: none;
            cursor: pointer;
            transition: all 0.2s ease;
        }
        
        .khmer-time-footer .btn-secondary {
            background: #6c757d;
            color: white;
        }
        
        .khmer-time-footer .btn-secondary:hover {
            background: #5a6268;
        }
        
        .khmer-time-footer .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        
        .khmer-time-footer .btn-primary:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(102, 126, 234, 0.3);
        }
        
        /* Responsive */
        @media (max-width: 480px) {
            .khmer-time-content {
                width: 95%;
                margin: 10px;
            }
            
            .khmer-time-selectors {
                gap: 10px;
            }
            
            .khmer-time-selector select {
                width: 70px;
                padding: 8px;
                font-size: 14px;
            }
            
            .khmer-time-separator {
                font-size: 20px;
            }
        }
        </style>
    @endonce
@endpush
